<?php
/**
 * The plugin activation class.
 *
 * @since      	1.1.0
 * @since  		1.5 Moved into /inc
 * @package    	LiteSpeed_Cache
 * @subpackage 	LiteSpeed_Cache/inc
 * @author     	LiteSpeed Technologies <info@litespeedtech.com>
 */
class LiteSpeed_Cache_Activation
{
	const NETWORK_TRANSIENT_COUNT = 'lscwp_network_count' ;

	/**
	 * The activation hook callback.
	 *
	 * Attempts to set up the advanced cache file. If it fails for any reason,
	 * the plugin will not activate.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public static function register_activation()
	{
		$count = 0 ;
		! defined( 'LSCWP_LOG_TAG' ) && define( 'LSCWP_LOG_TAG', 'Activate_' . get_current_blog_id() ) ;

		self::try_copy_advanced_cache() ;
		LiteSpeed_Cache_Config::wp_cache_var_setter( true ) ;

		if ( is_multisite() ) {
			$count = self::get_network_count() ;
			if ( $count !== false ) {
				$count = intval( $count ) + 1 ;
				set_site_transient( self::NETWORK_TRANSIENT_COUNT, $count, DAY_IN_SECONDS ) ;
			}
		}
		do_action( 'litespeed_cache_api_load_thirdparty' ) ;
		LiteSpeed_Cache_Config::get_instance()->plugin_activation( $count ) ;
		LiteSpeed_Cache_Admin_Report::get_instance()->generate_environment_report() ;

		if ( defined( 'LSCWP_PLUGIN_NAME' ) ) {
			set_transient( LiteSpeed_Cache::WHM_TRANSIENT, LiteSpeed_Cache::WHM_TRANSIENT_VAL ) ;
		}

		// Register crawler cron task
		LiteSpeed_Cache_Task::update() ;
	}

	/**
	 * Uninstall plugin
	 * @since 1.1.0
	 */
	public static function uninstall_litespeed_cache()
	{
		LiteSpeed_Cache_Task::clear() ;
		LiteSpeed_Cache_Admin_Rules::get_instance()->clear_rules( true ) ;
		delete_option( LiteSpeed_Cache_Config::OPTION_NAME ) ;
		if ( is_multisite() ) {
			delete_site_option( LiteSpeed_Cache_Config::OPTION_NAME ) ;
		}
	}

	/**
	 * Get the blog ids for the network. Accepts function arguments.
	 *
	 * Will use wp_get_sites for WP versions less than 4.6
	 *
	 * @since 1.0.12
	 * @access public
	 * @param array $args Arguments to pass into get_sites/wp_get_sites.
	 * @return array The array of blog ids.
	 */
	public static function get_network_ids( $args = array() )
	{
		global $wp_version ;
		if ( version_compare( $wp_version, '4.6', '<' ) ) {
			$blogs = wp_get_sites( $args ) ;
			if ( ! empty( $blogs ) ) {
				foreach ( $blogs as $key => $blog ) {
					$blogs[ $key ] = $blog[ 'blog_id' ] ;
				}
			}
		}
		else {
			$args[ 'fields' ] = 'ids' ;
			$blogs = get_sites( $args ) ;
		}
		return $blogs ;
	}

	/**
	 * Gets the count of active litespeed cache plugins on multisite.
	 *
	 * @since 1.0.12
	 * @access private
	 * @return mixed The count on success, false on failure.
	 */
	private static function get_network_count()
	{
		$count = get_site_transient( self::NETWORK_TRANSIENT_COUNT ) ;
		if ( $count !== false ) {
			return intval( $count ) ;
		}
		// need to update
		$default = array() ;
		$count = 0 ;

		$sites = self::get_network_ids( array( 'deleted' => 0 ) ) ;
		if ( empty( $sites ) ) {
			return false ;
		}

		foreach ( $sites as $site ) {
			$bid = is_object( $site ) && property_exists( $site, 'blog_id' ) ? $site->blog_id : $site ;
			$plugins = get_blog_option( $bid , 'active_plugins', $default ) ;
			if ( in_array( LSCWP_BASENAME, $plugins, true ) ) {
				$count++ ;
			}
		}
		if ( is_plugin_active_for_network( LSCWP_BASENAME ) ) {
			$count++ ;
		}
		return $count ;
	}

	/**
	 * Is this deactivate call the last active installation on the multisite
	 * network?
	 *
	 * @since 1.0.12
	 * @access private
	 * @return bool True if yes, false otherwise.
	 */
	private static function is_deactivate_last()
	{
		$count = self::get_network_count() ;
		if ( $count === false ) {
			return false ;
		}
		if ( $count !== 1 ) {
			// Not deactivating the last one.
			$count-- ;
			set_site_transient( self::NETWORK_TRANSIENT_COUNT, $count, DAY_IN_SECONDS ) ;
			return false ;
		}

		delete_site_transient( self::NETWORK_TRANSIENT_COUNT ) ;
		return true ;
	}

	/**
	 * The deactivation hook callback.
	 *
	 * Initializes all clean up functionalities.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public static function register_deactivation()
	{
		LiteSpeed_Cache_Task::clear() ;

		! defined( 'LSCWP_LOG_TAG' ) && define( 'LSCWP_LOG_TAG', 'Deactivate_' . get_current_blog_id() ) ;

		LiteSpeed_Cache_Purge::purge_all() ;

		if ( is_multisite() ) {

			if ( ! self::is_deactivate_last() ) {
				if ( is_network_admin() ) {
					// Still other activated subsite left, set .htaccess with only CacheLookUp
					LiteSpeed_Cache_Admin_Rules::get_instance()->insert_wrapper() ;
				}
				return ;
			}
		}

		$adv_cache_path = LSCWP_CONTENT_DIR . '/advanced-cache.php' ;
		// this file can be generated by other cache plugin like w3tc, we only delete our own file
		if ( file_exists( $adv_cache_path ) && is_writable( $adv_cache_path ) ) {
			if ( strpos( file_get_contents( $adv_cache_path ), 'LSCACHE_ADV_CACHE' ) !== false ) {
				unlink( $adv_cache_path ) ;
			}
			else {
				error_log(' Keep advanced-cache.php as it belongs to other plugins' ) ;
			}
		}
		else {
			error_log( 'Failed to remove advanced-cache.php, file does not exist or is not writable!' ) ;
		}

		/**
		 * Remove object cache file if is us
		 * @since  1.8.2
		 */
		LiteSpeed_Cache_Object::get_instance()->del_file() ;

		if ( ! LiteSpeed_Cache_Config::wp_cache_var_setter( false ) ) {
			error_log('In wp-config.php: WP_CACHE could not be set to false during deactivation!')  ;
		}

		LiteSpeed_Cache_Admin_Rules::get_instance()->clear_rules( true ) ;

		// delete in case it's not deleted prior to deactivation.
		self::dismiss_whm() ;
	}

	/**
	 * Try to copy our advanced-cache.php file to the wordpress directory.
	 *
	 * @since 1.0.11
	 * @access public
	 * @return boolean True on success, false on failure.
	 */
	public static function try_copy_advanced_cache()
	{
		$adv_cache_path = LSCWP_CONTENT_DIR . '/advanced-cache.php' ;
		if ( file_exists( $adv_cache_path ) && ( filesize( $adv_cache_path ) !== 0 || ! is_writable( $adv_cache_path ) ) ) {
			return false ;
		}

		copy( LSCWP_DIR . 'includes/advanced-cache.php', $adv_cache_path ) ;
		include( $adv_cache_path ) ;
		$ret = defined( 'LSCACHE_ADV_CACHE' ) ;
		return $ret ;
	}

	/**
	 * Delete whm transient msg tag
	 *
	 * @since 1.1.1
	 * @access public
	 */
	public static function dismiss_whm()
	{
		delete_transient( LiteSpeed_Cache::WHM_TRANSIENT ) ;
	}


}
